<?php
/**
 * Plugin Name: Technical SEO & Performance Optimizer
 * Description: Comprehensive SEO and performance optimization tool for WordPress with broken link detection, image optimization, and database cleanup.
 * Version: 1.0.4
 * Author: Felix
 * Author URI: https://example.com
 * License: GPL v2 or later
 * Text Domain: seo-optimizer
 */

if (!defined('ABSPATH')) exit;

define('SEO_OPT_VERSION', '1.0.4');
define('SEO_OPT_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SEO_OPT_PLUGIN_URL', plugin_dir_url(__FILE__));

class SEO_Performance_Optimizer {
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp_ajax_seo_opt_scan_links', array($this, 'ajax_scan_links'));
        add_action('wp_ajax_seo_opt_fix_links', array($this, 'ajax_fix_links'));
        add_action('wp_ajax_seo_opt_optimize_images', array($this, 'ajax_optimize_images'));
        add_action('wp_ajax_seo_opt_cleanup_database', array($this, 'ajax_cleanup_database'));
        add_action('wp_ajax_seo_opt_get_stats', array($this, 'ajax_get_stats'));
        add_action('wp_ajax_seo_opt_check_support', array($this, 'ajax_check_support'));
        add_filter('wp_handle_upload', array($this, 'auto_convert_uploaded_image'));
        register_activation_hook(__FILE__, array($this, 'activate'));
    }
    
    public function activate() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        $table_name = $wpdb->prefix . 'seo_broken_links';
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            url varchar(500) NOT NULL,
            post_id bigint(20) DEFAULT NULL,
            post_title text,
            link_text varchar(255),
            status_code int(3),
            found_date datetime DEFAULT CURRENT_TIMESTAMP,
            fixed tinyint(1) DEFAULT 0,
            PRIMARY KEY (id)
        ) $charset_collate;";
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        add_option('seo_opt_auto_convert', 'webp');
        add_option('seo_opt_image_quality', 85);
    }
    
    public function add_admin_menu() {
        add_menu_page('SEO Optimizer', 'SEO Optimizer', 'manage_options', 'seo-optimizer', array($this, 'render_main_page'), 'dashicons-performance', 30);
    }
    
    public function enqueue_admin_scripts($hook) {
        if ('toplevel_page_seo-optimizer' !== $hook) return;
        wp_enqueue_style('seo-opt-admin', SEO_OPT_PLUGIN_URL . 'assets/admin.css', array(), SEO_OPT_VERSION);
        wp_enqueue_script('seo-opt-admin', SEO_OPT_PLUGIN_URL . 'assets/admin.js', array('jquery'), SEO_OPT_VERSION, true);
        wp_localize_script('seo-opt-admin', 'seoOptAjax', array('ajax_url' => admin_url('admin-ajax.php'), 'nonce' => wp_create_nonce('seo_opt_nonce')));
    }
    
    public function register_settings() {
        register_setting('seo_opt_settings', 'seo_opt_auto_convert');
        register_setting('seo_opt_settings', 'seo_opt_image_quality');
    }
    
    public function render_main_page() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'seo_broken_links';
        $broken_links = $wpdb->get_results("SELECT * FROM $table_name WHERE fixed = 0");
        
        // Check server support
        $webp_support = $this->check_image_format_support('webp');
        $avif_support = $this->check_image_format_support('avif');
        $gd_installed = extension_loaded('gd') && function_exists('gd_info');
        ?>
        <div class="wrap seo-optimizer-wrap">
            <h1><span class="dashicons dashicons-performance"></span> Technical SEO & Performance Optimizer</h1>
            <div class="seo-opt-dashboard">
                <div class="seo-opt-stats">
                    <div class="stat-card">
                        <div class="stat-icon"><span class="dashicons dashicons-warning"></span></div>
                        <div class="stat-content"><h3 id="broken-links-count">0</h3><p>Broken Links</p></div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon"><span class="dashicons dashicons-images-alt2"></span></div>
                        <div class="stat-content"><h3 id="images-count">0</h3><p>Unoptimized Images</p></div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon"><span class="dashicons dashicons-database"></span></div>
                        <div class="stat-content"><h3 id="db-size">0 MB</h3><p>Database Size</p></div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon"><span class="dashicons dashicons-trash"></span></div>
                        <div class="stat-content"><h3 id="cleanup-items">0</h3><p>Cleanup Items</p></div>
                    </div>
                </div>
                
                <div class="seo-opt-tabs">
                    <button class="tab-button active" data-tab="broken-links"><span class="dashicons dashicons-admin-links"></span> Broken Links</button>
                    <button class="tab-button" data-tab="image-optimization"><span class="dashicons dashicons-format-image"></span> Image Optimization</button>
                    <button class="tab-button" data-tab="database-cleanup"><span class="dashicons dashicons-database-remove"></span> Database Cleanup</button>
                    <button class="tab-button" data-tab="settings"><span class="dashicons dashicons-admin-settings"></span> Settings</button>
                </div>
                
                <div class="tab-content active" id="broken-links">
                    <div class="seo-opt-section">
                        <h2>Broken Link Detection & Fixing</h2>
                        <div class="action-buttons">
                            <button id="scan-links-btn" class="button button-primary"><span class="dashicons dashicons-search"></span> Scan for Broken Links</button>
                        </div>
                        <div id="scan-progress" class="progress-container" style="display:none;">
                            <div class="progress-bar"><div class="progress-fill"></div></div>
                            <p class="progress-text">Scanning...</p>
                        </div>
                        <div id="broken-links-table" class="links-table-container">
                            <?php if (count($broken_links) > 0): ?>
                                <div class="bulk-actions-bar">
                                    <label><input type="checkbox" id="select-all-links" /> Select All</label>
                                    <select id="bulk-action-select">
                                        <option value="">Bulk Actions</option>
                                        <option value="delete">Delete Links</option>
                                        <option value="homepage">Redirect to Homepage</option>
                                    </select>
                                    <button id="apply-bulk-action" class="button">Apply</button>
                                </div>
                                <table class="wp-list-table widefat fixed striped">
                                    <thead>
                                        <tr>
                                            <td class="check-column"><input type="checkbox" /></td>
                                            <th>Broken URL</th><th>Found in Post</th><th>Link Text</th><th>Status Code</th><th>Found Date</th><th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody id="broken-links-tbody">
                                        <?php foreach ($broken_links as $link): ?>
                                            <tr data-link-id="<?php echo esc_attr($link->id); ?>">
                                                <th class="check-column"><input type="checkbox" class="link-checkbox" value="<?php echo esc_attr($link->id); ?>" /></th>
                                                <td class="broken-url"><?php echo esc_html($link->url); ?></td>
                                                <td><?php if ($link->post_id): ?><a href="<?php echo get_edit_post_link($link->post_id); ?>" target="_blank"><?php echo esc_html($link->post_title); ?></a><?php else: ?>N/A<?php endif; ?></td>
                                                <td><?php echo esc_html($link->link_text); ?></td>
                                                <td><span class="status-code status-<?php echo esc_attr($link->status_code); ?>"><?php echo esc_html($link->status_code); ?></span></td>
                                                <td><?php echo esc_html($link->found_date); ?></td>
                                                <td>
                                                    <button class="button button-small fix-link-btn" data-link-id="<?php echo esc_attr($link->id); ?>" data-action="delete">Delete</button>
                                                    <button class="button button-small fix-link-btn" data-link-id="<?php echo esc_attr($link->id); ?>" data-action="homepage">To Homepage</button>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            <?php else: ?>
                                <div class="no-issues"><span class="dashicons dashicons-yes-alt"></span><p>No broken links found. Click "Scan for Broken Links" to check your site.</p></div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                
                <div class="tab-content" id="image-optimization">
                    <div class="seo-opt-section">
                        <h2>Image Optimization</h2>
                        
                        <!-- Server Support Notice -->
                        <div class="notice notice-info" style="padding: 15px; margin: 20px 0;">
                            <h3 style="margin-top: 0;">Server Image Support:</h3>
                            <ul style="margin: 10px 0 10px 20px;">
                                <li><strong>GD Library:</strong> <?php echo $gd_installed ? '<span style="color: green;">✓ Installed</span>' : '<span style="color: red;">✗ Not Installed</span>'; ?></li>
                                <li><strong>WebP Support:</strong> <?php echo $webp_support ? '<span style="color: green;">✓ Enabled</span>' : '<span style="color: red;">✗ Not Available</span>'; ?></li>
                                <li><strong>AVIF Support:</strong> <?php echo $avif_support ? '<span style="color: green;">✓ Enabled</span>' : '<span style="color: red;">✗ Not Available</span>'; ?></li>
                            </ul>
                            <?php if (!$gd_installed): ?>
                                <p style="color: red; font-weight: bold;">⚠️ GD Library is not installed. Image optimization requires GD Library. Please contact your hosting provider.</p>
                            <?php elseif (!$webp_support && !$avif_support): ?>
                                <p style="color: red; font-weight: bold;">⚠️ Your server does not support WebP or AVIF. Please contact your hosting provider to enable these formats.</p>
                            <?php endif; ?>
                        </div>
                        
                        <p>Convert images to WebP or AVIF format for better performance. New uploads will be automatically converted.</p>
                        <div class="action-buttons">
                            <button id="optimize-images-btn" class="button button-primary" <?php echo (!$webp_support && !$avif_support) ? 'disabled' : ''; ?>>
                                <span class="dashicons dashicons-images-alt2"></span> Optimize All Images
                            </button>
                            <button id="check-support-btn" class="button" style="margin-left: 10px;">
                                <span class="dashicons dashicons-info"></span> Check Detailed Support
                            </button>
                        </div>
                        <div id="image-progress" class="progress-container" style="display:none;">
                            <div class="progress-bar"><div class="progress-fill"></div></div>
                            <p class="progress-text">Optimizing images...</p>
                        </div>
                        <div id="image-results" class="results-container"></div>
                        <div class="settings-box">
                            <h3>Image Settings</h3>
                            <table class="form-table">
                                <tr>
                                    <th>Auto-Convert Format</th>
                                    <td>
                                        <select name="seo_opt_auto_convert" id="auto-convert-format">
                                            <option value="webp" <?php selected(get_option('seo_opt_auto_convert'), 'webp'); ?> <?php echo !$webp_support ? 'disabled' : ''; ?>>
                                                WebP <?php echo !$webp_support ? '(Not Supported)' : ''; ?>
                                            </option>
                                            <option value="avif" <?php selected(get_option('seo_opt_auto_convert'), 'avif'); ?> <?php echo !$avif_support ? 'disabled' : ''; ?>>
                                                AVIF <?php echo !$avif_support ? '(Not Supported)' : ''; ?>
                                            </option>
                                        </select>
                                    </td>
                                </tr>
                                <tr>
                                    <th>Image Quality</th>
                                    <td>
                                        <input type="number" name="seo_opt_image_quality" id="image-quality" value="<?php echo esc_attr(get_option('seo_opt_image_quality', 85)); ?>" min="1" max="100" />
                                        <p class="description">Quality percentage (1-100). Higher = better quality but larger file size.</p>
                                    </td>
                                </tr>
                            </table>
                            <button id="save-image-settings" class="button button-primary">Save Settings</button>
                        </div>
                    </div>
                </div>
                
                <div class="tab-content" id="database-cleanup">
                    <div class="seo-opt-section">
                        <h2>Database Cleanup & Resource Monitoring</h2>
                        <div class="cleanup-options">
                            <h3>Select Items to Clean</h3>
                            <label><input type="checkbox" class="cleanup-checkbox" value="revisions" checked /> Post Revisions</label>
                            <label><input type="checkbox" class="cleanup-checkbox" value="autodrafts" checked /> Auto-Drafts</label>
                            <label><input type="checkbox" class="cleanup-checkbox" value="trash" checked /> Trashed Posts</label>
                            <label><input type="checkbox" class="cleanup-checkbox" value="spam" checked /> Spam Comments</label>
                            <label><input type="checkbox" class="cleanup-checkbox" value="trashed_comments" checked /> Trashed Comments</label>
                            <label><input type="checkbox" class="cleanup-checkbox" value="transients" checked /> Expired Transients</label>
                            <label><input type="checkbox" class="cleanup-checkbox" value="orphaned_meta" /> Orphaned Post Meta</label>
                        </div>
                        <div class="action-buttons">
                            <button id="cleanup-database-btn" class="button button-primary"><span class="dashicons dashicons-database-remove"></span> Clean Database</button>
                        </div>
                        <div id="cleanup-results" class="results-container"></div>
                        <div class="resource-monitor">
                            <h3>Resource Monitor</h3>
                            <div class="monitor-grid">
                                <div class="monitor-item"><strong>Database Size:</strong><span id="monitor-db-size">Calculating...</span></div>
                                <div class="monitor-item"><strong>Post Revisions:</strong><span id="monitor-revisions">0</span></div>
                                <div class="monitor-item"><strong>Auto-Drafts:</strong><span id="monitor-autodrafts">0</span></div>
                                <div class="monitor-item"><strong>Trashed Items:</strong><span id="monitor-trash">0</span></div>
                                <div class="monitor-item"><strong>Spam Comments:</strong><span id="monitor-spam">0</span></div>
                                <div class="monitor-item"><strong>Expired Transients:</strong><span id="monitor-transients">0</span></div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="tab-content" id="settings">
                    <div class="seo-opt-section">
                        <h2>Plugin Settings</h2>
                        <div class="settings-box">
                            <h3>General Settings</h3>
                            <p>Configure your SEO & Performance optimization preferences.</p>
                            <table class="form-table">
                                <tr><th>Plugin Version</th><td><?php echo SEO_OPT_VERSION; ?></td></tr>
                                <tr><th>Last Scan</th><td id="last-scan-date">Never</td></tr>
                            </table>
                        </div>
                        <div class="settings-box">
                            <h3>About</h3>
                            <p>Technical SEO & Performance Optimizer helps you maintain a healthy, fast-loading WordPress website by detecting and fixing broken links, optimizing images, and cleaning up your database.</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Check if a specific image format is supported
     */
    private function check_image_format_support($format) {
        if (!extension_loaded('gd') || !function_exists('gd_info')) {
            return false;
        }
        
        $gd_info = gd_info();
        
        switch ($format) {
            case 'webp':
                return (function_exists('imagewebp') && !empty($gd_info['WebP Support']));
            case 'avif':
                return (function_exists('imageavif') && !empty($gd_info['AVIF Support']));
            case 'jpeg':
            case 'jpg':
                return !empty($gd_info['JPEG Support']);
            case 'png':
                return !empty($gd_info['PNG Support']);
            case 'gif':
                return !empty($gd_info['GIF Read Support']);
            default:
                return false;
        }
    }
    
    public function ajax_check_support() {
        check_ajax_referer('seo_opt_nonce', 'nonce');
        
        $gd_info = function_exists('gd_info') ? gd_info() : array();
        
        $support_info = array(
            'php_version' => PHP_VERSION,
            'gd_installed' => extension_loaded('gd') && function_exists('gd_info'),
            'webp_support' => $this->check_image_format_support('webp'),
            'avif_support' => $this->check_image_format_support('avif'),
            'gd_info' => $gd_info,
            'memory_limit' => ini_get('memory_limit'),
            'max_execution_time' => ini_get('max_execution_time'),
            'upload_max_filesize' => ini_get('upload_max_filesize'),
            'post_max_size' => ini_get('post_max_size')
        );
        
        wp_send_json_success($support_info);
    }
    
    public function ajax_scan_links() {
        check_ajax_referer('seo_opt_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        global $wpdb;
        $table_name = $wpdb->prefix . 'seo_broken_links';
        $wpdb->query("TRUNCATE TABLE $table_name");
        $posts = get_posts(array('post_type' => array('post', 'page'), 'post_status' => 'publish', 'numberposts' => -1));
        $broken_links = array();
        foreach ($posts as $post) {
            $content = $post->post_content;
            preg_match_all('/<a[^>]+href=["\'](.*?)["\'][^>]*>(.*?)<\/a>/i', $content, $matches);
            if (!empty($matches[1])) {
                foreach ($matches[1] as $index => $url) {
                    if (strpos($url, '#') === 0 || strpos($url, 'mailto:') === 0) continue;
                    $link_text = strip_tags($matches[2][$index]);
                    $status_code = $this->check_url_status($url);
                    if ($status_code >= 400 || $status_code === 0) {
                        $wpdb->insert($table_name, array('url' => $url, 'post_id' => $post->ID, 'post_title' => $post->post_title, 'link_text' => substr($link_text, 0, 255), 'status_code' => $status_code, 'fixed' => 0));
                        $broken_links[] = array('id' => $wpdb->insert_id, 'url' => $url, 'post_id' => $post->ID, 'post_title' => $post->post_title, 'link_text' => $link_text, 'status_code' => $status_code);
                    }
                }
            }
        }
        update_option('seo_opt_last_scan', current_time('mysql'));
        wp_send_json_success(array('broken_links' => $broken_links, 'count' => count($broken_links)));
    }
    
    private function check_url_status($url) {
        $response = wp_remote_head($url, array('timeout' => 10, 'redirection' => 5, 'sslverify' => false));
        if (is_wp_error($response)) return 0;
        return wp_remote_retrieve_response_code($response);
    }
    
    public function ajax_fix_links() {
        check_ajax_referer('seo_opt_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        $link_ids = isset($_POST['link_ids']) ? array_map('intval', $_POST['link_ids']) : array();
        $action = isset($_POST['fix_action']) ? sanitize_text_field($_POST['fix_action']) : '';
        if (empty($link_ids) || empty($action)) wp_send_json_error('Invalid parameters');
        global $wpdb;
        $table_name = $wpdb->prefix . 'seo_broken_links';
        $fixed_count = 0;
        foreach ($link_ids as $link_id) {
            $link = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $link_id));
            if (!$link || !$link->post_id) continue;
            $post = get_post($link->post_id);
            if (!$post) continue;
            $content = $post->post_content;
            if ($action === 'delete') {
                $pattern = '/<a[^>]+href=["\']' . preg_quote($link->url, '/') . '["\'][^>]*>(.*?)<\/a>/i';
                $content = preg_replace($pattern, '$1', $content);
            } elseif ($action === 'homepage') {
                $homepage = home_url('/');
                $pattern = '/href=["\']' . preg_quote($link->url, '/') . '["\']/i';
                $content = preg_replace($pattern, 'href="' . $homepage . '"', $content);
            }
            wp_update_post(array('ID' => $link->post_id, 'post_content' => $content));
            $wpdb->update($table_name, array('fixed' => 1), array('id' => $link_id));
            $fixed_count++;
        }
        wp_send_json_success(array('fixed_count' => $fixed_count));
    }
    
    public function ajax_optimize_images() {
        check_ajax_referer('seo_opt_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        // Increase memory limit and execution time
        @ini_set('memory_limit', '512M');
        @set_time_limit(300);
        
        $format = get_option('seo_opt_auto_convert', 'webp');
        $quality = intval(get_option('seo_opt_image_quality', 85));
        
        // Validate quality
        if ($quality < 1 || $quality > 100) {
            $quality = 85;
        }
        
        // Check if format is supported
        if (!$this->check_image_format_support($format)) {
            wp_send_json_error(ucfirst($format) . ' format is not supported on your server. Please check your GD Library configuration.');
        }
        
        $args = array(
            'post_type' => 'attachment',
            'post_mime_type' => array('image/jpeg', 'image/jpg', 'image/png'),
            'post_status' => 'inherit',
            'numberposts' => -1,
            'fields' => 'ids'
        );
        
        $image_ids = get_posts($args);
        
        if (empty($image_ids)) {
            wp_send_json_success(array(
                'optimized' => 0,
                'skipped' => 0,
                'total' => 0,
                'errors' => array(),
                'details' => array('No images found to optimize.')
            ));
        }
        
        $optimized = 0;
        $skipped = 0;
        $errors = array();
        $details = array();
        
        foreach ($image_ids as $image_id) {
            $file_path = get_attached_file($image_id);
            
            if (!$file_path || !file_exists($file_path)) {
                $skipped++;
                $details[] = "Skipped ID {$image_id}: File not found";
                continue;
            }
            
            $current_ext = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
            $file_size_before = filesize($file_path);
            
            // Skip if already in target format
            if ($current_ext === $format) {
                $skipped++;
                $details[] = "Skipped ID {$image_id}: Already in {$format} format";
                continue;
            }
            
            // Only process supported formats
            if (!in_array($current_ext, array('jpg', 'jpeg', 'png'))) {
                $skipped++;
                $details[] = "Skipped ID {$image_id}: Unsupported format ({$current_ext})";
                continue;
            }
            
            $result = $this->convert_image_to_format($file_path, $format, $quality, $image_id);
            
            if ($result['success']) {
                $optimized++;
                $file_size_after = file_exists($result['new_file']) ? filesize($result['new_file']) : 0;
                $saved = $file_size_before - $file_size_after;
                $saved_percent = $file_size_before > 0 ? round(($saved / $file_size_before) * 100, 2) : 0;
                $details[] = "✓ ID {$image_id}: Converted successfully (Saved: " . size_format($saved) . " / {$saved_percent}%)";
            } else {
                $skipped++;
                $errors[] = "ID {$image_id}: " . $result['error'];
                $details[] = "✗ ID {$image_id}: " . $result['error'];
            }
            
            // Free memory after each image
            if (function_exists('gc_collect_cycles')) {
                gc_collect_cycles();
            }
        }
        
        wp_send_json_success(array(
            'optimized' => $optimized,
            'skipped' => $skipped,
            'total' => count($image_ids),
            'errors' => $errors,
            'details' => $details
        ));
    }
    
    /**
     * Convert image to specified format with proper error handling
     * FIXED: Better MIME type detection and image resource creation
     */
    private function convert_image_to_format($file_path, $format, $quality, $attachment_id) {
        // Validate inputs
        if (!file_exists($file_path)) {
            return array('success' => false, 'error' => 'File does not exist');
        }
        
        if (!is_readable($file_path)) {
            return array('success' => false, 'error' => 'File is not readable');
        }
        
        // Check GD library
        if (!extension_loaded('gd') || !function_exists('gd_info')) {
            return array('success' => false, 'error' => 'GD Library not installed');
        }
        
        // Check format support
        if (!$this->check_image_format_support($format)) {
            return array('success' => false, 'error' => ucfirst($format) . ' format not supported');
        }
        
        // Get MIME type using WordPress function (more reliable)
        $mime_type = wp_check_filetype($file_path);
        $mime = $mime_type['type'];
        
        // Also try to get image info
        $image_info = @getimagesize($file_path);
        if ($image_info === false) {
            return array('success' => false, 'error' => 'Invalid or corrupted image');
        }
        
        list($width, $height, $image_type) = $image_info;
        
        // Check if image is too large
        $max_pixels = 25000000; // 25 megapixels
        if (($width * $height) > $max_pixels) {
            return array('success' => false, 'error' => 'Image too large (max 25MP)');
        }
        
        // Check memory before processing (rough estimate)
        $memory_needed = ($width * $height * 4 * 1.8); // 4 bytes per pixel, 1.8x for processing
        $memory_limit = ini_get('memory_limit');
        $memory_limit_bytes = $this->convert_to_bytes($memory_limit);
        $memory_used = memory_get_usage(true);
        
        if ($memory_limit_bytes > 0 && ($memory_used + $memory_needed) > ($memory_limit_bytes * 0.9)) {
            return array('success' => false, 'error' => 'Not enough memory (Image: ' . round($memory_needed / 1024 / 1024, 2) . 'MB needed)');
        }
        
        // Create image resource - try different methods
        $source_image = null;
        $ext = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
        
        // Use image type from getimagesize as primary method (most reliable)
        switch ($image_type) {
            case IMAGETYPE_JPEG:
                $source_image = @imagecreatefromjpeg($file_path);
                break;
            case IMAGETYPE_PNG:
                $source_image = @imagecreatefrompng($file_path);
                if ($source_image) {
                    // Preserve transparency for PNG
                    imagealphablending($source_image, false);
                    imagesavealpha($source_image, true);
                }
                break;
            case IMAGETYPE_GIF:
                $source_image = @imagecreatefromgif($file_path);
                break;
            default:
                // Fallback: try based on file extension
                if ($ext === 'jpg' || $ext === 'jpeg') {
                    $source_image = @imagecreatefromjpeg($file_path);
                } elseif ($ext === 'png') {
                    $source_image = @imagecreatefrompng($file_path);
                    if ($source_image) {
                        imagealphablending($source_image, false);
                        imagesavealpha($source_image, true);
                    }
                } elseif ($ext === 'gif') {
                    $source_image = @imagecreatefromgif($file_path);
                }
                break;
        }
        
        // Check if image resource was created successfully
        if (!$source_image) {
            return array('success' => false, 'error' => 'Failed to create image resource (Type: ' . $image_type . ', Ext: ' . $ext . ')');
        }
        
        // Verify it's a valid GD resource
        if (!is_resource($source_image) && !($source_image instanceof \GdImage)) {
            return array('success' => false, 'error' => 'Invalid image resource created');
        }
        
        // Generate new filename
        $path_parts = pathinfo($file_path);
        $new_file = $path_parts['dirname'] . '/' . $path_parts['filename'] . '.' . $format;
        
        // Check if directory is writable
        if (!is_writable($path_parts['dirname'])) {
            imagedestroy($source_image);
            return array('success' => false, 'error' => 'Directory not writable');
        }
        
        // Save converted image
        $save_success = false;
        
        if ($format === 'webp') {
            $save_success = @imagewebp($source_image, $new_file, $quality);
        } elseif ($format === 'avif') {
            $save_success = @imageavif($source_image, $new_file, $quality);
        }
        
        // Free memory
        imagedestroy($source_image);
        
        if (!$save_success) {
            return array('success' => false, 'error' => 'Failed to save converted image');
        }
        
        if (!file_exists($new_file)) {
            return array('success' => false, 'error' => 'Converted file not created');
        }
        
        // Update WordPress metadata
        try {
            // Update file path
            update_attached_file($attachment_id, $new_file);
            
            // Update MIME type
            wp_update_post(array(
                'ID' => $attachment_id,
                'post_mime_type' => 'image/' . $format
            ));
            
            // Regenerate thumbnails
            require_once(ABSPATH . 'wp-admin/includes/image.php');
            $metadata = wp_generate_attachment_metadata($attachment_id, $new_file);
            wp_update_attachment_metadata($attachment_id, $metadata);
            
            // Delete old file and thumbnails
            if ($file_path !== $new_file && file_exists($file_path)) {
                @unlink($file_path);
                
                // Delete old thumbnails
                $old_thumbs = glob($path_parts['dirname'] . '/' . $path_parts['filename'] . '-*.' . $path_parts['extension']);
                if (is_array($old_thumbs)) {
                    foreach ($old_thumbs as $old_thumb) {
                        @unlink($old_thumb);
                    }
                }
            }
            
            return array('success' => true, 'new_file' => $new_file);
            
        } catch (Exception $e) {
            // Rollback: delete new file if metadata update fails
            if (file_exists($new_file)) {
                @unlink($new_file);
            }
            return array('success' => false, 'error' => 'Metadata update failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Auto-convert uploaded images
     */
    public function auto_convert_uploaded_image($upload) {
        // Check if it's an image
        if (empty($upload['file']) || empty($upload['type']) || strpos($upload['type'], 'image/') !== 0) {
            return $upload;
        }
        
        $format = get_option('seo_opt_auto_convert', 'webp');
        $quality = intval(get_option('seo_opt_image_quality', 85));
        
        // Check if format is supported
        if (!$this->check_image_format_support($format)) {
            return $upload;
        }
        
        $file_path = $upload['file'];
        $current_ext = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
        
        // Only convert supported formats
        if (!in_array($current_ext, array('jpg', 'jpeg', 'png'))) {
            return $upload;
        }
        
        // Skip if already in target format
        if ($current_ext === $format) {
            return $upload;
        }
        
        // Convert the image
        $result = $this->convert_image_simple($file_path, $format, $quality);
        
        if ($result['success']) {
            $new_file = $result['new_file'];
            $upload['file'] = $new_file;
            $upload['url'] = str_replace(basename($file_path), basename($new_file), $upload['url']);
            $upload['type'] = 'image/' . $format;
            
            // Delete original file
            if (file_exists($file_path) && $file_path !== $new_file) {
                @unlink($file_path);
            }
        }
        
        return $upload;
    }
    
    /**
     * Simple image conversion without metadata update (for uploads)
     * FIXED: Better error handling and image resource creation
     */
    private function convert_image_simple($file_path, $format, $quality) {
        if (!file_exists($file_path) || !is_readable($file_path)) {
            return array('success' => false, 'error' => 'File not accessible');
        }
        
        if (!$this->check_image_format_support($format)) {
            return array('success' => false, 'error' => 'Format not supported');
        }
        
        $image_info = @getimagesize($file_path);
        if ($image_info === false) {
            return array('success' => false, 'error' => 'Invalid image');
        }
        
        $image_type = $image_info[2];
        $ext = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
        $source_image = null;
        
        // Create image resource based on image type (most reliable)
        switch ($image_type) {
            case IMAGETYPE_JPEG:
                $source_image = @imagecreatefromjpeg($file_path);
                break;
            case IMAGETYPE_PNG:
                $source_image = @imagecreatefrompng($file_path);
                if ($source_image) {
                    imagealphablending($source_image, false);
                    imagesavealpha($source_image, true);
                }
                break;
            case IMAGETYPE_GIF:
                $source_image = @imagecreatefromgif($file_path);
                break;
            default:
                // Fallback to extension
                if ($ext === 'jpg' || $ext === 'jpeg') {
                    $source_image = @imagecreatefromjpeg($file_path);
                } elseif ($ext === 'png') {
                    $source_image = @imagecreatefrompng($file_path);
                    if ($source_image) {
                        imagealphablending($source_image, false);
                        imagesavealpha($source_image, true);
                    }
                } elseif ($ext === 'gif') {
                    $source_image = @imagecreatefromgif($file_path);
                }
                break;
        }
        
        if (!$source_image) {
            return array('success' => false, 'error' => 'Failed to create resource (Type: ' . $image_type . ')');
        }
        
        $path_parts = pathinfo($file_path);
        $new_file = $path_parts['dirname'] . '/' . $path_parts['filename'] . '.' . $format;
        
        $save_success = false;
        if ($format === 'webp') {
            $save_success = @imagewebp($source_image, $new_file, $quality);
        } elseif ($format === 'avif') {
            $save_success = @imageavif($source_image, $new_file, $quality);
        }
        
        imagedestroy($source_image);
        
        if (!$save_success || !file_exists($new_file)) {
            return array('success' => false, 'error' => 'Save failed');
        }
        
        return array('success' => true, 'new_file' => $new_file);
    }
    
    /**
     * Convert PHP memory notation to bytes
     */
    private function convert_to_bytes($size_str) {
        if (empty($size_str)) {
            return 0;
        }
        
        $size_str = trim($size_str);
        $last = strtolower($size_str[strlen($size_str)-1]);
        $size_str = (int) $size_str;
        
        switch($last) {
            case 'g':
                $size_str *= 1024;
            case 'm':
                $size_str *= 1024;
            case 'k':
                $size_str *= 1024;
        }
        
        return $size_str;
    }
    
    public function ajax_cleanup_database() {
        check_ajax_referer('seo_opt_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        global $wpdb;
        $cleanup_items = isset($_POST['cleanup_items']) ? $_POST['cleanup_items'] : array();
        $results = array();
        foreach ($cleanup_items as $item) {
            switch ($item) {
                case 'revisions': $results['revisions'] = $wpdb->query("DELETE FROM $wpdb->posts WHERE post_type = 'revision'"); break;
                case 'autodrafts': $results['autodrafts'] = $wpdb->query("DELETE FROM $wpdb->posts WHERE post_status = 'auto-draft'"); break;
                case 'trash': $results['trash'] = $wpdb->query("DELETE FROM $wpdb->posts WHERE post_status = 'trash'"); break;
                case 'spam': $results['spam'] = $wpdb->query("DELETE FROM $wpdb->comments WHERE comment_approved = 'spam'"); break;
                case 'trashed_comments': $results['trashed_comments'] = $wpdb->query("DELETE FROM $wpdb->comments WHERE comment_approved = 'trash'"); break;
                case 'transients': $results['transients'] = $wpdb->query("DELETE FROM $wpdb->options WHERE option_name LIKE '_transient_%'"); break;
                case 'orphaned_meta': $results['orphaned_meta'] = $wpdb->query("DELETE pm FROM $wpdb->postmeta pm LEFT JOIN $wpdb->posts p ON p.ID = pm.post_id WHERE p.ID IS NULL"); break;
            }
        }
        $wpdb->query("OPTIMIZE TABLE $wpdb->posts");
        $wpdb->query("OPTIMIZE TABLE $wpdb->postmeta");
        $wpdb->query("OPTIMIZE TABLE $wpdb->comments");
        $wpdb->query("OPTIMIZE TABLE $wpdb->options");
        wp_send_json_success($results);
    }
    
    public function ajax_get_stats() {
        check_ajax_referer('seo_opt_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        global $wpdb;
        $db_size = $wpdb->get_var("SELECT SUM(data_length + index_length) / 1024 / 1024 FROM information_schema.TABLES WHERE table_schema = '" . DB_NAME . "'");
        $table_name = $wpdb->prefix . 'seo_broken_links';
        $broken_links = $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE fixed = 0");
        
        $format = get_option('seo_opt_auto_convert', 'webp');
        
        // Count unoptimized images
        $args = array(
            'post_type' => 'attachment',
            'post_mime_type' => array('image/jpeg', 'image/jpg', 'image/png'),
            'post_status' => 'inherit',
            'numberposts' => -1,
            'fields' => 'ids'
        );
        
        $image_ids = get_posts($args);
        $unoptimized = 0;
        
        foreach ($image_ids as $image_id) {
            $file_path = get_attached_file($image_id);
            if ($file_path && file_exists($file_path)) {
                $ext = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
                if ($ext !== $format && in_array($ext, array('jpg', 'jpeg', 'png'))) {
                    $unoptimized++;
                }
            }
        }
        
        $revisions = $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->posts WHERE post_type = 'revision'");
        $autodrafts = $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->posts WHERE post_status = 'auto-draft'");
        $trash = $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->posts WHERE post_status = 'trash'");
        $spam = $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->comments WHERE comment_approved = 'spam'");
        $trashed_comments = $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->comments WHERE comment_approved = 'trash'");
        $transients = $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->options WHERE option_name LIKE '_transient_%'");
        $cleanup_total = $revisions + $autodrafts + $trash + $spam + $trashed_comments + $transients;
        $last_scan = get_option('seo_opt_last_scan', 'Never');
        wp_send_json_success(array('broken_links' => intval($broken_links), 'images' => $unoptimized, 'db_size' => round($db_size, 2), 'cleanup_items' => $cleanup_total, 'revisions' => intval($revisions), 'autodrafts' => intval($autodrafts), 'trash' => intval($trash), 'spam' => intval($spam), 'trashed_comments' => intval($trashed_comments), 'transients' => intval($transients), 'last_scan' => $last_scan));
    }
}

SEO_Performance_Optimizer::get_instance();